import { notFound } from "next/navigation"
import Image from "next/image"
import Link from "next/link"
import { Header } from "@/components/header"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Calendar, User, ArrowLeft, ExternalLink } from "lucide-react"

async function getBlogPost(slug: string) {
  try {
    const response = await fetch(
      `${process.env.NEXT_PUBLIC_SITE_URL || "http://localhost:3000"}/api/blog/posts/${slug}`,
      {
        next: { revalidate: 3600 },
      },
    )

    if (!response.ok) {
      return null
    }

    const data = await response.json()
    return data.post
  } catch (error) {
    console.error("Error fetching blog post:", error)
    return null
  }
}

export default async function BlogPostPage({
  params,
}: {
  params: { slug: string }
}) {
  const post = await getBlogPost(params.slug)

  if (!post) {
    notFound()
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "long",
      day: "numeric",
    })
  }

  return (
    <div className="min-h-screen bg-background">
      <Header />

      <div className="container mx-auto px-4 py-8">
        <div className="max-w-4xl mx-auto">
          {/* Back Button */}
          <Button variant="ghost" asChild className="mb-6">
            <Link href="/blog">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Blog
            </Link>
          </Button>

          {/* Article Header */}
          <article className="space-y-8">
            <header className="space-y-6">
              {/* Categories */}
              <div className="flex flex-wrap gap-2">
                {post.blog_post_categories.map((category: any, index: number) => (
                  <Badge key={index} variant="secondary">
                    {category.blog_categories.name}
                  </Badge>
                ))}
                {post.movie_title && <Badge variant="outline">Movie: {post.movie_title}</Badge>}
              </div>

              {/* Title */}
              <h1 className="text-4xl font-bold leading-tight">{post.title}</h1>

              {/* Meta Information */}
              <div className="flex items-center text-muted-foreground space-x-6">
                <div className="flex items-center">
                  <User className="h-4 w-4 mr-2" />
                  <span>{post.profiles.email.split("@")[0]}</span>
                </div>
                <div className="flex items-center">
                  <Calendar className="h-4 w-4 mr-2" />
                  <span>{formatDate(post.published_at)}</span>
                </div>
              </div>

              {/* Featured Image */}
              {post.featured_image && (
                <div className="relative aspect-video rounded-lg overflow-hidden">
                  <Image
                    src={post.featured_image || "/placeholder.svg"}
                    alt={post.title}
                    fill
                    className="object-cover"
                  />
                </div>
              )}

              {/* Excerpt */}
              {post.excerpt && (
                <div className="text-xl text-muted-foreground leading-relaxed border-l-4 border-primary pl-6">
                  {post.excerpt}
                </div>
              )}
            </header>

            {/* Article Content */}
            <div className="prose prose-gray dark:prose-invert max-w-none">
              <div dangerouslySetInnerHTML={{ __html: post.content }} />
            </div>

            {/* Movie Link */}
            {post.movie_id && (
              <Card className="border-primary/20 bg-primary/5">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <h3 className="text-lg font-semibold mb-2">Watch {post.movie_title}</h3>
                      <p className="text-muted-foreground">
                        Get more details, watch trailers, and find where to stream this movie.
                      </p>
                    </div>
                    <Button asChild>
                      <Link href={`/movie/${post.movie_id}`}>
                        <ExternalLink className="h-4 w-4 mr-2" />
                        View Movie
                      </Link>
                    </Button>
                  </div>
                </CardContent>
              </Card>
            )}
          </article>
        </div>
      </div>
    </div>
  )
}

export async function generateMetadata({
  params,
}: {
  params: { slug: string }
}) {
  const post = await getBlogPost(params.slug)

  if (!post) {
    return {
      title: "Blog Post Not Found",
    }
  }

  return {
    title: `${post.title} - MoviesWeb Blog`,
    description: post.excerpt || post.title,
    openGraph: {
      title: post.title,
      description: post.excerpt || post.title,
      images: post.featured_image ? [post.featured_image] : [],
    },
  }
}
